<?php

namespace Psphost\SimplePay\Controller\EndPayment;

use Psphost\SimplePay\Common\PspOperationException;
use Psphost\SimplePay\Common\PspOrderIdFormatter;
use Psphost\SimplePay\Common\PspRefundProcessor;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\View\Result\PageFactory;
use Magento\Framework\App\Request\InvalidRequestException;
use Psphost\SimplePay\Signer;
use Magento\Sales\Api\CreditmemoRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Service\InvoiceService;
use Magento\Sales\Model\Order\Creditmemo;
use Magento\Store\Model\ScopeInterface;


class Index extends Action implements CsrfAwareActionInterface
{
    /**
     * @var OrderRepositoryInterface
     */
    protected $orderRepository;

    /**
     * @var InvoiceService
     */
    protected $invoiceService;

    /**
     * @var CreditmemoRepositoryInterface
     */
    protected $creditmemoRepository;

    /**
     * @var Signer
     */
    protected $signer;

    /**
     * @var PageFactory
     */
    protected $pageFactory;

    /**
     * @var string|null
     */
    protected $currentOrderLink;

    /**
     * @var JsonFactory
     */
    protected $resultJsonFactory;

    /**
     * @param Context $context
     * @param PageFactory $pageFactory
     * @param OrderRepositoryInterface $orderRepository
     * @param CreditmemoRepositoryInterface $creditmemoRepository
     * @param InvoiceService $invoiceService
     * @param JsonFactory $resultJsonFactory
     */
    public function __construct(
        Context $context,
        OrderRepositoryInterface $orderRepository,
        CreditmemoRepositoryInterface $creditmemoRepository,
        InvoiceService $invoiceService,
        PageFactory $pageFactory,
        JsonFactory $resultJsonFactory
    )
    {
        parent::__construct($context);
        $this->orderRepository = $orderRepository;
        $this->creditmemoRepository = $creditmemoRepository;
        $this->invoiceService = $invoiceService;
        $this->pageFactory = $pageFactory;
        $this->resultJsonFactory = $resultJsonFactory;
        $this->signer = new Signer();
    }

    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        return null;
    }

    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }

    public function execute()
    {
        $body = file_get_contents('php://input');

        $configIsTest = $this->signer->getConfig()['isTest'];

        if (!empty($body)) {
            $this->processRefundCallback($body);
        }

        $resultPage = $this->pageFactory->create();
        $resultPage->getLayout()->initMessages();

        if ($configIsTest && $this->getRequest()->getParam('test')) {
            return $this->checkTestOrder($resultPage);
        }

        if ($this->getRequest()->getParam('on_success')) {
            $message = __('Your payment is being processed by psphost');

            $orderId = $this->getRequest()->getParam('order_id');
            list($order, $_) = $this->getOrder($orderId);
            if ($order && $order->getStatus() === \Magento\Sales\Model\Order::STATE_PAYMENT_REVIEW) {
                $message = __('Payment successfully processed by psphost');
            }

            $resultPage->getLayout()->getBlock('endpayment')->setOrderMessage($message);
            $resultPage->getLayout()->getBlock('endpayment')->setOrderLink($this->currentOrderLink);
            return $resultPage;
        }

        return $this->checkRealOrder($resultPage, $body);
    }

    protected function processRefundCallback($data)
    {
        /** @var \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig */
        $scopeConfig = $this->_objectManager->get('\Magento\Framework\App\Config\ScopeConfigInterface');
        $storeScope = ScopeInterface::SCOPE_STORE;

        $isTest = $scopeConfig->getValue(Signer::XML_IS_TEST, $storeScope);
        $projectId = $scopeConfig->getValue(Signer::XML_PROJECT_ID, $storeScope);
        $salt = $scopeConfig->getValue(Signer::XML_SALT, $storeScope);

        $refundProcessor = new PspRefundProcessor(
            $isTest ? Signer::TEST_PROJECT_ID : $projectId,
            $isTest ? Signer::TEST_PROJECT_KEY : $salt,
            $isTest ? Signer::CMS_PREFIX : ''
        );

        try {
            $pspCallbackResult = $refundProcessor->processCallback($data);
        } catch (PspOperationException $ex) {
            return; // not a refund operation, go back to order confirmation
        } catch (\Exception $e) {
            http_response_code(400);
            die($e->getMessage());
        }

        $resource = $this->_objectManager->get('Magento\Framework\App\ResourceConnection');
        /** @var \Magento\Framework\App\ResourceConnection $resource */
        $connection = $resource->getConnection();
        $tableName = $resource->getTableName('sales_creditmemo_comment');

        $comment = sprintf(Signer::REFUND_ID_CONTAINING_COMMENT, $pspCallbackResult->getRefundExternalId());

        $sql = "SELECT entity_id, parent_id FROM $tableName WHERE comment = '$comment'";

        $result = $connection->fetchRow($sql);
        if (!$result) {
            http_response_code(400);
            die('Unknown operation identifier.');
        }
        try {
            /** @var Creditmemo $cm */
            $cm = $this->creditmemoRepository->get($result['parent_id']);
            $order = $cm->getOrder();
        } catch (\Exception $e) {
            http_response_code(400);
            die('Unable to find a refund request.');
        }

        if (!$connection->delete($tableName, 'entity_id = ' . $result['entity_id'])) {
            http_response_code(400);
            die('Unable to process callback.');
        }

        $now = new \DateTime();
        $refundId = $cm->getId();
        if ($pspCallbackResult->isSuccess()) {
            $cm->setState(Creditmemo::STATE_REFUNDED);
            $cm->addComment('Money-back request #' . $refundId . ' was successfully processed at ' . $now->format('d.m.Y H:i:s'));
            $order->setStatus(PspRefundProcessor::ORDER_STATUS)->save();
        } else {
            $cm->setState(Creditmemo::STATE_CANCELED);
            $cm->addComment('Money-back request #' . $refundId . ' was declined at ' . $now->format('d.m.Y H:i:s') . '. Reason: ' . $pspCallbackResult->getDescription());
        }
        try {
            $cm->save();
        } catch (\Exception $e) {
            http_response_code(400);
            die('Unable to process callback.');
        }

        http_response_code(200);
        die();
    }

    protected function checkTestOrder($resultPage) {
        $orderId = $this->getRequest()->getParam('order_id');
        $paymentStatus = $this->getRequest()->getParam('status');

        list ($order, $message) = $this->getOrder($orderId);
        if (!$order) {
            $resultPage->getLayout()->getBlock('endpayment')->setOrderMessage($message);
            return $resultPage;
        }

        $message = __('Payment failed to be processed by psphost');

        if ($paymentStatus === 'success') {
            $message = __('Payment successfully processed by psphost');
            $paymentId = time();
            $this->updateOrderStatus($order, $message, $paymentId);
        }

        $resultPage->getLayout()->getBlock('endpayment')->setOrderMessage($message);
        $resultPage->getLayout()->getBlock('endpayment')->setOrderLink($this->currentOrderLink);
        return $resultPage;
    }

    protected function checkRealOrder($resultPage, $body)
    {
        $bodyData = json_decode($body, true);

        $message = __('Signature failed verification');
        if (is_array($bodyData) && $this->signer->checkSignature($bodyData)) {
            $orderId = $bodyData['payment']['id'];
            $orderId = PspOrderIdFormatter::removeOrderPrefix($orderId, Signer::CMS_PREFIX);

            $paymentStatus = $bodyData['payment']['status'];

            list ($order, $message) = $this->getOrder($orderId);
            if (!$order) {
                $resultPage->getLayout()->getBlock('endpayment')->setOrderMessage($message);
                return $resultPage;
            }

            $message = __('Payment failed to be processed by psphost');

            if ($paymentStatus === 'success') {
                $paymentId = $bodyData['operation']['id'];

                $message = __('Payment successfully processed by psphost');
                $this->updateOrderStatus($order, $message, $paymentId);
            } else {
                $order->addStatusToHistory($order->getStatus(), $message);
                $order->save();
            }
        }
        $resultPage->getLayout()->getBlock('endpayment')->setOrderMessage($message);
        return $resultPage;
    }

    /**
     * @param $orderId
     * @return array
     */
    protected function getOrder($orderId)
    {
        $message = '';
        $order = null;

        try {
            $order = $this->orderRepository->get($orderId);

            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
            $customerSession = $objectManager->get('Magento\Customer\Model\Session');
            if($customerSession->isLoggedIn()) {
                $this->currentOrderLink = '/sales/order/view/order_id/' . $orderId . '/';
            }
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            $message = __('Order does not exist');
        } catch (\Magento\Framework\Exception\InputException $e) {
            $message = __('Order does not exist, no order_id is provided');
        }
        return [$order, $message];
    }

    /**
     * @param Order $order
     * @param string $message
     * @param int $paymentId
     * @throws \Exception
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function updateOrderStatus($order, $message, $paymentId)
    {
        $order->setState(\Magento\Sales\Model\Order::STATE_PROCESSING);
        $order->setStatus(\Magento\Sales\Model\Order::STATE_PROCESSING);
        $order->addStatusToHistory($order->getStatus(), $message);
        //$order->save();

        $invoice = $this->invoiceService->prepareInvoice($order);
        $invoice->register();
        $invoice->setTransactionId($paymentId);
        $invoice->setState(\Magento\Sales\Model\Order\Invoice::STATE_PAID);
        $invoice->save();

        /** @var Order\Payment $payment */
        $payment = $order->getPayment();
        $payment->pay($invoice);

        $order->setTotalPaid($order->getTotalPaid() + $payment->getAmountPaid());
        $order->setBaseTotalPaid($order->getBaseTotalPaid() + $payment->getAmountPaid());
        $order->save();
    }
}