<?php

class Signer
{
    /**
     * @var bool
     */
    protected $isTest;

    /**
     * @var string|int
     */
    protected $projectId;

    /**
     * @var string
     */
    protected $secretKey;

    /**
     * @var string
     */
    protected $ppLanguage;

    /**
     * @var string
     */
    protected $ppCurrency;

    /**
     * @var string
     */
    protected $additionalParameters;

    /**
     * @var string
     */
    protected $baseUrl;

    const PSPHOST_TEST_PROJECT_ID = 147626;
    const PSPHOST_TEST_SECRET_KEY = '11b1f2370a8306af3b798dc3f38a431f1afb0be37455458321100de9a68be9ce2c15b4a400c951f5cc636da0aa4b3dca4004dd8377b1f5b5fbf64c42e259bdd2';

    const INTERFACE_TYPE = 17;

    const CMS_PREFIX = 'presta';

    /**
     * @return array
     */
    public static function getInterfaceType()
    {
        return [
            'id' => self::INTERFACE_TYPE,
        ];
    }

    /**
     * Signer constructor.
     * @param $isTest
     * @param $projectId
     * @param $secretKey
     * @param $ppLanguage
     * @param $ppCurrency
     * @param $additionalParameters
     * @param $baseUrl
     */
    public function __construct($isTest, $projectId, $secretKey, $ppLanguage, $ppCurrency, $additionalParameters, $baseUrl)
    {
        $this->isTest = $isTest;
        $this->projectId = $projectId;
        $this->secretKey = $secretKey;
        $this->ppLanguage = $ppLanguage;
        $this->ppCurrency = $ppCurrency;
        $this->additionalParameters = $additionalParameters;
        $this->baseUrl = $baseUrl;
    }

    /**
     * Return PaymentPage host
     *
     * @return string
     */
    public static function getPaymentPageHost()
    {
        $host = getenv('PAYMENTPAGE_HOST');

        return \is_string($host) ? $host : 'paymentpage.psphost.com';
    }

    /**
     * @return array
     */
    public function getConfig()
    {
        $isTest = $this->isTest;
        $projectId = $this->projectId;
        $salt = $this->secretKey;
        $ppCurrency = $this->ppCurrency;
        $ppLanguage = $this->ppLanguage;
        $additionalParameters = $this->additionalParameters;

        return compact(
            'isTest',
            'projectId',
            'salt',
            'ppCurrency',
            'ppLanguage',
            'additionalParameters'
        );
    }

    /**
     * @return string
     */
    public function getCardRedirectUrl($cart)
    {
        $config = $this->getConfig();

        $successUrl = $this->getSuccessUrl($cart, $config);
        $failUrl = str_replace('PLACEHOLDER', 'failpayment', $this->baseUrl);

        $urlData = $this->createUrlData($cart, $config, $successUrl, $failUrl);

        if (!empty($config['additionalParameters'])) {
            $additionalData = [];
            parse_str($config['additionalParameters'], $additionalData);
            $urlData = array_merge($urlData, $additionalData);
        }

        $urlData['signature'] = $this->signData($urlData, []);
        $urlArgs = http_build_query($urlData, '', '&');

        return sprintf('https://%s/payment?%s', self::getPaymentPageHost(), $urlArgs);
    }

    /**
     * @param $cart
     * @param array $config
     * @param $successUrl
     * @param $failUrl
     * @return array
     */
    protected function createUrlData($cart, array $config, $successUrl, $failUrl)
    {
        $paymentAmount = $cart->getOrderTotal() * 100;
        $orderId = Order::getOrderByCartId($cart->id);
        $paymentId = $orderId;

        if ($config['isTest']) {
            require_once(__DIR__ . '/common/PsphostOrderIdFormatter.php');
            $paymentId = PsphostOrderIdFormatter::addOrderPrefix($paymentId, self::CMS_PREFIX);
        }

        $urlParams = [
            'project_id' => $config['isTest'] ? self::PSPHOST_TEST_PROJECT_ID : $config['projectId'],
            'payment_amount' => $paymentAmount,
            'payment_id' => $paymentId,
            'payment_currency' => $config['ppCurrency'],
            'language_code' => strtolower($config['ppLanguage']),
            'merchant_success_url' => $successUrl,
            'merchant_fail_url' => $failUrl,
            'interface_type' => json_encode(self::getInterfaceType()),
        ];
        $urlParams['merchant_fail_enabled'] = 2;
        $urlParams['merchant_success_enabled'] = 2;

        if($cart->id_customer){
            $customer = new Customer((int)$cart->id_customer);
            if ($customer && !$customer->isGuest()) {
                $urlParams['customer_id'] = $cart->id_customer;
            }
        }

        return $urlParams;
    }

    /**
     * @param $cart
     * @param array $config
     * @return string
     */
    private function getSuccessUrl($cart, array $config) {
        $orderId = Order::getOrderByCartId($cart->id);
        $prefix = str_replace('PLACEHOLDER', 'successpayment', $this->baseUrl);
        if ($config['isTest']) {
            return $prefix . '?' . http_build_query([
                    'order_id' => $orderId,
                    'test' => 1
                ]);
        }
        return $prefix . '?' . http_build_query(['order_id' => $orderId]);
    }


    /**
     * Get parameters to sign
     * @param array $params
     * @param array $ignoreParamKeys
     * @param int $currentLevel
     * @param string $prefix
     * @return array
     */
    private function getParamsToSign(
        array $params,
        array $ignoreParamKeys = [],
        $currentLevel = 1,
        $prefix = ''
    )
    {
        $paramsToSign = [];
        foreach ($params as $key => $value) {
            if ((in_array($key, $ignoreParamKeys) && $currentLevel == 1)) {
                continue;
            }
            $paramKey = ($prefix ? $prefix . ':' : '') . $key;
            if (is_array($value)) {
                if ($currentLevel >= 3) {
                    $paramsToSign[$paramKey] = (string)$paramKey.':';
                } else {
                    $subArray = self::getParamsToSign($value, $ignoreParamKeys, $currentLevel + 1, $paramKey);
                    $paramsToSign = array_merge($paramsToSign, $subArray);
                }
            } else {
                if (is_bool($value)) {
                    $value = $value ? '1' : '0';
                } else {
                    $value = (string)$value;
                }
                $paramsToSign[$paramKey] = (string)$paramKey.':'.$value;
            }
        }
        if ($currentLevel == 1) {
            ksort($paramsToSign, SORT_NATURAL);
        }
        return $paramsToSign;
    }

    private function signData(array $data, $skipParams) {
        $config = $this->getConfig();
        $paramsToSign = $this->getParamsToSign($data, $skipParams);
        $stringToSign = $this->getStringToSign($paramsToSign);
        $secretKey = $config['isTest'] ? self::PSPHOST_TEST_SECRET_KEY : $config['salt'];
        return base64_encode(hash_hmac('sha512', $stringToSign, $secretKey, true));
    }

    private function getStringToSign(array $paramsToSign)
    {
        return implode(';', $paramsToSign);
    }

    public function checkSignature(array $data) {
        $signature = $data['signature'];
        unset($data['signature']);
        return $this->signData($data, []) === $signature;
    }
}
