<?php

namespace Psphost\SimplePay;

use Psphost\SimplePay\Common\PspOrderIdFormatter;

class Signer
{
    const XML_IS_TEST = 'payment/simplepay/testmode';
    const XML_PROJECT_ID = 'payment/simplepay/project_id';
    const XML_SALT = 'payment/simplepay/salt';
    const XML_PP_LANGUAGE = 'payment/simplepay/pp_language';
    const XML_PP_CURRENCY = 'payment/simplepay/pp_currency';
    const XML_PP_ADDITIONAL_PARAMS = 'payment/simplepay/pp_additional_params';

    const DEFAULT_HOST = 'paymentpage.psphost.com';
    const TEST_PROJECT_ID = 147626;
    const TEST_PROJECT_KEY = '11b1f2370a8306af3b798dc3f38a431f1afb0be37455458321100de9a68be9ce2c15b4a400c951f5cc636da0aa4b3dca4004dd8377b1f5b5fbf64c42e259bdd2';

    const INTERFACE_TYPE = 13;

    const CMS_PREFIX = 'mag';

    const REFUND_ID_CONTAINING_COMMENT = 'Refund request #%s was created';

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Checkout\Model\Session
     */
    protected $checkoutSession;

    /**
     * @var \Magento\Framework\UrlInterface
     */
    protected $urlBuilder;

    /**
     * Signer constructor.
     */
    public function __construct()
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();

        $this->scopeConfig = $objectManager->get('\Magento\Framework\App\Config\ScopeConfigInterface');
        $this->checkoutSession = $objectManager->get('\Magento\Checkout\Model\Session');
        $this->urlBuilder = $objectManager->get('Magento\Framework\UrlInterface');
    }

    /**
     * @return array
     */
    public static function getInterfaceType()
    {
        return [
            'id' => self::INTERFACE_TYPE,
        ];
    }

    /**
     * @return array
     */
    public function getConfig()
    {
        $storeScope = \Magento\Store\Model\ScopeInterface::SCOPE_STORE;

        $isTest = $this->scopeConfig->getValue(self::XML_IS_TEST, $storeScope);
        $projectId = $this->scopeConfig->getValue(self::XML_PROJECT_ID, $storeScope);
        $salt = $this->scopeConfig->getValue(self::XML_SALT, $storeScope);
        $ppLanguage = $this->scopeConfig->getValue(self::XML_PP_LANGUAGE, $storeScope);
        $ppCurrency = $this->scopeConfig->getValue(self::XML_PP_CURRENCY, $storeScope);
        $ppAdditionalParams = $this->scopeConfig->getValue(self::XML_PP_ADDITIONAL_PARAMS, $storeScope);

        return compact(
            'isTest',
            'projectId',
            'salt',
            'ppCurrency',
            'ppLanguage',
            'ppAdditionalParams'
        );
    }

    /**
     * @return string
     */
    public function getOrderRedirectUrl(\Magento\Sales\Model\Order $order)
    {
        $config = $this->getConfig();

        $successUrl = $this->getSuccessUrl($order, $config);
        $failUrl = $this->urlBuilder->getUrl('psphost/failpayment/index');

        $urlData = $this->createUrlData($order, $config, $successUrl, $failUrl);

        if (!empty($config['ppAdditionalParams'])) {
            $additionalData = [];
            parse_str($config['ppAdditionalParams'], $additionalData);
            $urlData = array_merge($urlData, $additionalData);
        }

        $urlData['merchant_success_enabled'] = 2;
        $urlData['merchant_fail_enabled'] = 2;

        $urlData['signature'] = $this->signData($urlData, []);
        $urlArgs = http_build_query($urlData, '', '&');

        return sprintf(
            'https://%s/payment?%s',
            $this->getPaymentPageHost(),
            $urlArgs
        );
    }

    /**
     * @param array $config
     * @param $successUrl
     * @param $failUrl
     * @return array
     */
    protected function createUrlData(\Magento\Sales\Model\Order $order, array $config, $successUrl, $failUrl)
    {
        $payment_id = $order->getId();
        if ($config['isTest']) {
            $payment_id = PspOrderIdFormatter::addOrderPrefix($payment_id, self::CMS_PREFIX);
        }

        $result = [
            'project_id' => $config['isTest'] ? self::TEST_PROJECT_ID : $config['projectId'],
            'payment_amount' => $order->getTotalDue() * 100,
            'payment_id' => $payment_id,
            'payment_currency' => $config['ppCurrency'],
            'language_code' => $config['ppLanguage'],
            'merchant_success_url' => $successUrl,
            'merchant_fail_url' => $failUrl,
            'interface_type' => json_encode(self::getInterfaceType()),
        ];

        if ($order->getCustomerId()) {
            $result['customer_id'] = $order->getCustomerId();
        }

        return $result;
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @param array $config
     * @return string
     */
    private function getSuccessUrl(\Magento\Sales\Model\Order $order, array $config)
    {
        $prefix = $this->urlBuilder->getUrl('psphost/endpayment/index');
        if ($config['isTest']) {
            return $prefix . '?' . http_build_query([
                    'order_id' => $order->getId(),
                    'test' => 1,
                    'on_success' => 1,
                    'status' => 'success'
                ]);
        }
        return $prefix . '?' . http_build_query(['on_success' => 1, 'order_id' => $order->getId()]);
    }

    /**
     * Get parameters to sign
     *
     * @param array $params
     * @param array $ignoreParamKeys
     * @param int $currentLevel
     * @param string $prefix
     * @return array
     */
    private function getParamsToSign(
        array $params,
        array $ignoreParamKeys = [],
        $currentLevel = 1,
        $prefix = ''
    ) {
        $paramsToSign = [];
        foreach ($params as $key => $value) {
            if ((in_array($key, $ignoreParamKeys) && $currentLevel == 1)) {
                continue;
            }
            $paramKey = ($prefix ? $prefix . ':' : '') . $key;
            if (is_array($value)) {
                if ($currentLevel >= 3) {
                    $paramsToSign[$paramKey] = (string)$paramKey.':';
                } else {
                    $subArray = self::getParamsToSign($value, $ignoreParamKeys, $currentLevel + 1, $paramKey);
                    $paramsToSign = array_merge($paramsToSign, $subArray);
                }
            } else {
                if (is_bool($value)) {
                    $value = $value ? '1' : '0';
                } else {
                    $value = (string)$value;
                }
                $paramsToSign[$paramKey] = (string)$paramKey.':'.$value;
            }
        }
        if ($currentLevel == 1) {
            ksort($paramsToSign, SORT_NATURAL);
        }
        return $paramsToSign;
    }

    private function signData(array $data, $skipParams)
    {
        $config = $this->getConfig();
        $paramsToSign = $this->getParamsToSign($data, $skipParams);
        $stringToSign = $this->getStringToSign($paramsToSign);
        $secretKey = $config['isTest'] ? self::TEST_PROJECT_KEY : $config['salt'];
        return base64_encode(hash_hmac('sha512', $stringToSign, $secretKey, true));
    }

    private function getStringToSign(array $paramsToSign)
    {
        return implode(';', $paramsToSign);
    }

    public function checkSignature(array $data)
    {
        $signature = $data['signature'];
        unset($data['signature']);
        return $this->signData($data, []) === $signature;
    }

    private function getPaymentPageHost()
    {
        $host = getenv('PAYMENTPAGE_HOST');
        return is_string($host) ? $host : self::DEFAULT_HOST;
    }
}
