<?php

namespace Psphost\SimplePay\Common;

use Psphost\SimplePay\Signer;
use Exception;

class PspRefundProcessor
{
    const ORDER_STATUS = 'partial_refund';

    const PSPHOST_GATE_PROTO = 'https';
    const PSPHOST_GATE_HOST = 'api.psphost.com';
    const GATE_REFUND_ENDPOINT = '/v2/payment/card/refund';

    /**
     * @var int
     */
    private $projectId;

    /**
     * @var PspSigner
     */
    private $signer;

    /**
     * @var string
     */
    private $paymentPrefix;

    /**
     * @param $projectId
     * @param string $secretKey
     * @param string $paymentPrefix
     */
    public function __construct($projectId, $secretKey, $paymentPrefix = '')
    {
        $this->projectId = (int)$projectId;
        $this->signer = new PspSigner($secretKey);
        $this->paymentPrefix = $paymentPrefix;
    }


    /**
     * @param string $orderId
     * @param float $amount
     * @param string $currency
     * @param string $reason
     * @return PspRefundResult
     * @throws Exception
     */
    public function processRefund($orderId, $amount, $currency, $reason = '')
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->getGateEndpoint());
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $post = [
            'general' => [
                'project_id' => $this->projectId,
                'payment_id' => $this->paymentPrefix
                    ? PspOrderIdFormatter::addOrderPrefix($orderId, $this->paymentPrefix)
                    : $orderId
            ],
            'payment' => [
                'amount' => round($amount * 100), //
                'currency' => $currency,
                'description' => $reason
            ],
            'interface_type' => Signer::getInterfaceType(),
        ];

        $post['general']['signature'] = $this->signer->getSignature($post);

        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json'));
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($post));
        $out = curl_exec($ch);
        $httpStatus = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $data = json_decode($out, true);
        if ($data === null) {
            throw new Exception('Malformed response');
        }

        if ($httpStatus != 200) {
            return new PspRefundResult($orderId, null, $data['status'], $data['message']);
        }

        return new PspRefundResult($orderId, $data['request_id'], $data['status']);
    }

    /**
     * @param string $rawData
     * @return PspRefundResult
     * @throws Exception
     */
    public function processCallback($rawData)
    {
        $data = json_decode($rawData, true);
        if ($data === null) {
            throw new Exception('Malformed callback data.');
        }
        if (empty($data['operation']) || empty($data['operation']['type']) || $data['operation']['type'] !== 'refund') {
            throw new PspOperationException('Invalid or missed operation type, expected "refund".');
        }
        if (!$this->signer->checkSignature($data)) {
            throw new Exception('Wrong data signature.');
        }
        if (empty($data['operation']['status'])) {
            throw new Exception('Empty "status" field in callback data.');
        }
        $status = $data['operation']['status'];
        if (!in_array($status, ['success', 'decline'])) {
            throw new Exception('Received status is not final.');
        }

        if (empty($data['payment']) || empty($data['payment']['id'])) {
            throw new Exception('Missed "payment.id" field in callback data.');
        }
        $orderId = PspOrderIdFormatter::removeOrderPrefix($data['payment']['id'], $this->paymentPrefix);
        if (empty($data['operation']['request_id'])) {
            throw new Exception('Empty "operation.request_id" field in callback data.');
        }
        $refundExternalId = $data['operation']['request_id'];

        $description = null;
        if (!empty($data['operation']['message'])) {
            $description = $data['operation']['message'];
        }

        return new PspRefundResult($orderId, $refundExternalId, $status, $description);
    }

    protected function getGateEndpoint()
    {
        $proto = getenv('PSPHOST_GATE_PROTO') ?? self::PSPHOST_GATE_PROTO;
        $host = getenv('PSPHOST_GATE_HOST') ?? self::PSPHOST_GATE_HOST;

        return $proto.'://'.$host.self::GATE_REFUND_ENDPOINT;
    }
}
