<?php

require_once __DIR__ . '/PsphostSigner.php';
require_once __DIR__ . '/PsphostRefundResult.php';
require_once __DIR__ . '/PsphostOperationException.php';
require_once __DIR__ . '/PsphostOrderIdFormatter.php';

class PsphostRefundProcessor
{
    const
        PSPHOST_GATE_PROTO = 'https',
        PSPHOST_GATE_HOST = 'api.psphost.ru',
        GATE_REFUND_ENDPOINT = '/v2/payment/card/refund'
    ;

    /**
     * @var int
     */
    private $projectId;

    /**
     * @var PsphostSigner
     */
    private $signer;

    /**
     * @var string
     */
    private $paymentPrefix;

    /**
     * @param $projectId
     * @param string $secretKey
     * @param string $paymentPrefix
     */
    public function __construct($projectId, $secretKey, $paymentPrefix = '')
    {
        $this->projectId = (int)$projectId;
        $this->signer = new PsphostSigner($secretKey);
        $this->paymentPrefix = $paymentPrefix;
    }


    /**
     * @param string $orderId
     * @param float $amount
     * @param string $currency
     * @param string $reason
     * @return PsphostRefundResult
     * @throws Exception
     */
    public function processRefund($orderId, $amount, $currency, $reason = '')
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, self::getGateEndpoint());
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $post = [
            'general' => [
                'project_id' => $this->projectId,
                'payment_id' => $this->paymentPrefix
                    ? PsphostOrderIdFormatter::addOrderPrefix($orderId, $this->paymentPrefix)
                    : $orderId
            ],
            'payment' => [
                'amount' => round($amount * 100), //
                'currency' => $currency,
                'description' => $reason
            ],
            'interface_type' => WC_Gateway_Psphost::getInterfaceType(),
        ];

        $post['general']['signature'] = $this->signer->getSignature($post);

        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json'));
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($post));
        $out = curl_exec($ch);
        $httpStatus = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $data = json_decode($out, true);
        if ($data === null) {
            throw new Exception('Malformed response');
        }

        if ($httpStatus != 200) {
            return (new PsphostRefundResult($orderId, null, $data['status'], $data['message']))
                ->setPaymentStatus($data['payment']['status']);
        }

        return (new PsphostRefundResult($orderId, $data['request_id'], $data['status']))
            ->setPaymentStatus($data['payment']['status']);
    }

    /**
     * @param array $data
     * @return PsphostRefundResult
     * @throws Exception
     */
    public function processCallback($data)
    {
        if ($data === null) {
            throw new Exception('Malformed callback data.');
        }
        if (empty($data['operation']) || empty($data['operation']['type']) || $data['operation']['type'] !== 'refund') {
            throw new PsphostOperationException('Invalid or missed operation type, expected "refund".');
        }
        if (!$this->signer->checkSignature($data)) {
            throw new Exception('Wrong data signature.');
        }
        if (empty($data['operation']['status'])) {
            throw new Exception('Empty "status" field in callback data.');
        }
        $paymentStatus   = $data['payment']['status'] ?? null;
        if (!in_array($paymentStatus, ['success', 'decline', 'refunded', 'partially refunded'])) {
            throw new Exception('Received status is not final.');
        }

        if (empty($data['payment']['id'])) {
            throw new Exception('Missed "payment.id" field in callback data.');
        }
        $orderId = PsphostOrderIdFormatter::removeOrderPrefix($data['payment']['id'], $this->paymentPrefix);
        if (empty($data['operation']['request_id'])) {
            throw new Exception('Empty "operation.request_id" field in callback data.');
        }
        $refundExternalId = $data['operation']['request_id'];
        $operationStatus  = $data['operation']['status'] ?? null;
        $description      = $data['operation']['message'] ?? null;

        return
            (new PsphostRefundResult($orderId, $refundExternalId, $operationStatus, $description))
                ->setPaymentStatus($paymentStatus);
    }

    /**
     * @return string
     */
    protected function getGateEndpoint()
    {
        $proto = getenv('PSPHOST_PROTO') ?? self::PSPHOST_GATE_PROTO;
        $host = getenv('PSPHOST_GATE_HOST') ?? self::PSPHOST_GATE_HOST;

        return $proto.'://'.$host.self::GATE_REFUND_ENDPOINT;
    }
}
