<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

require_once 'common/PsphostSigner.php';

class WC_Gateway_Psphost extends WC_Payment_Gateway {

    const TEST_PROJECT_ID = 147626;
    const TEST_PROJECT_KEY = '11b1f2370a8306af3b798dc3f38a431f1afb0be37455458321100de9a68be9ce2c15b4a400c951f5cc636da0aa4b3dca4004dd8377b1f5b5fbf64c42e259bdd2';

    const INTERFACE_TYPE = 18;

    const CMS_PREFIX = 'wp';

    public $supports = array(
        'products',
        'refunds'
    );

    /**
     * @var PsphostRefundProcessor
     */
    private $refund_processor;

    /**
     * @var PsphostSigner
     */
    private $psphost_signer;

    /**
     * @return array
     */
    public static function getInterfaceType()
    {
        return [
            'id' => self::INTERFACE_TYPE,
        ];
    }

    public function __construct() {
        $this->id = 'psphost';
        $this->has_fields = false;
        $this->method_title = __('Psphost', 'woocommerce');
        $this->method_description = __('Settings of psphost payment gateway', 'woocommerce');
        $this->form_fields = $this->init_form_fields();
        $this->init_settings();

        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');

        add_action('woocommerce_update_options_payment_gateways_'.$this->id, array($this, 'process_admin_options'));

        $testMode = $this->get_option('test') === 'yes';

        $secret_key = $testMode ? self::TEST_PROJECT_KEY : $this->get_option('salt');
        $this->psphost_signer = new PsphostSigner($secret_key);

        $project_id = $testMode ? self::TEST_PROJECT_ID : $this->get_option('project_id');

        require_once('common/PsphostRefundProcessor.php');
        $this->refund_processor = new PsphostRefundProcessor($project_id, $secret_key, $testMode ? self::CMS_PREFIX : '');
    }

    /**
     * @return string
     */
    public static function getProtocol() {
        $proto = getenv('PSPHOST_PROTO');

        return \is_string($proto) ? $proto : 'https';
    }

    /**
     * @return string
     */
    public static function getPaymentPageHost() {
        $host = getenv('PAYMENTPAGE_HOST');
        return \is_string($host) ? $host : 'paymentpage.psphost.ru';
    }

    public function init_form_fields() {
        $callback_url = add_query_arg('wc-api', 'WC_Gateway_Psphost', '/');
        $callback_url = get_site_url(null, $callback_url);

        return [
            'enabled' => [
                'title' => __('Enable Psphost', 'woocommerce'),
                'type' => 'checkbox',
                'default' => 'no'
            ],
            'title' => [
                'title' => __('Title', 'woocommerce'),
                'type' => 'text',
                'description' => __('Title which is shown to customer', 'woocommerce'),
                'desc_tip' => true,
                'default' => __('Payment via Psphost', 'woocommerce')
            ],
            'description' => [
                'title' => __('Description', 'woocommerce'),
                'type' => 'text',
                'description' => __('Description which is shown to customer', 'woocommerce'),
                'desc_tip' => true,
                'default' => __(
                    'You will be redirected to Psphost payment page. All data you enter in that page are secured',
                    'woocommerce'
                )
            ],
            'test' => [
                'title' => __('Test mode', 'woocommerce'),
                'type' => 'checkbox',
                'default' => 'no'
            ],
            'project_id' => [
                'title' => __('Project ID', 'woocommerce'),
                'description' => __('Your project ID you could get from Psphost helpdesk. Leave it blank if test mode'),
                'type' => 'number',
                'desc_tip' => true,
            ],
            'salt' => [
                'title' => __('Secret key', 'woocommerce'),
                'description' => __(
                    'Secret key which is using to sign payment request. You could get it from Psphost helpdesk',
                    'woocommerce'
                ),
                'type' => 'password',
                'desc_tip' => true
            ],
            'language' => [
                'title' => __('Language', 'woocommerce'),
                'description' => __('Language of payment page', 'woocommerce'),
                'type' => 'select',
                'desc_tip' => true,
                'options' => [
                    'en' => __('English', 'woocommerce'),
                    'zh' => __('Chinese', 'woocommerce'),
                ],
                'default' => 'en'
            ],
            'additional' => [
                'title' => __('Additional parameters', 'woocommerce'),
                'description' => __('It will be added to redirect link to Psphost payment page (parse_str format)'),
                'type' => 'text',
                'desc_tip' => true,
                'default' => ''
            ],
            'mode' => [
                'title' => __('Display mode', 'woocommerce'),
                'description' => __('Payment page display mode', 'woocommerce'),
                'type' => 'select',
                'desc_tip' => true,
                'options' => [
                    'redirect' => __('Redirect', 'woocommerce'),
                    'popup' => __('Popup', 'woocommerce'),
                    'iframe' => __('Iframe', 'woocommerce'),
                ],
                'default' => 'redirect'
            ],
            'callback' => [
                'title' => 'Callback endpoint',
                'type' => 'title',
                'description' => sprintf(
                    __('You should provide callback endpoint <strong>%s</strong> to Psphost helpdesk. It is required' .
                        ' to get information about payment\'s status', 'woocommerce'),
                    $callback_url
                )
            ]
        ];
    }

    public function process_payment($order_id) {
        global $woocommerce;
        $order = new WC_Order($order_id);

        $order->update_status('on-hold', __('Awaiting payment', 'woocommerce'));
        $order->reduce_order_stock();

        $woocommerce->cart->empty_cart();

        return array(
            'result' => 'success',
            'redirect' => $this->get_redirect_url($order)
        );
    }

    private function get_redirect_url(WC_Order $order) {
        $project_id = $this->get_option('test') === 'yes' ? self::TEST_PROJECT_ID : $this->get_option('project_id');

        $payment_id = $order->get_order_number();
        if ($this->get_option('test') === 'yes') {
            $payment_id = PsphostOrderIdFormatter::addOrderPrefix($payment_id, self::CMS_PREFIX);
        }

        $url_data = [
            'project_id' => $project_id,
            'payment_amount' => $order->get_total()*100,
            'payment_id' =>  $payment_id,
            'payment_currency' => get_woocommerce_currency(),
            'language_code' => $this->get_option('language'),
            'interface_type' => json_encode(self::getInterfaceType()),
        ];

        if ($this->get_option('mode') === 'iframe') {
            $url_data['target_element'] = 'psphost_iframe';
        }

        // Set strictly only if current customer is registered
        if ($customer_id = $order->get_customer_id()) {
            $url_data['customer_id'] = $customer_id;
        }

        $additional = $this->get_option('additional');
        if (!empty($additional)) {
            $additional_data = [];
            parse_str(wp_specialchars_decode($additional), $additional_data);
            $url_data = array_merge($url_data, $additional_data);
        }


        $url_data['signature'] = $this->psphost_signer->getSignature($url_data);
        $url_args = http_build_query($url_data, '', '&');

        return self::getProtocol().'://' . self::getPaymentPageHost() . '/payment?' . $url_args;
    }

    /**
     * @param array $bodyData
     */
    public function process_callback($bodyData)
    {
        if (is_array($bodyData) && $this->psphost_signer->checkSignature($bodyData)) {
            $order_id = $bodyData['payment']['id'];
            $order_id = PsphostOrderIdFormatter::removeOrderPrefix($order_id, self::CMS_PREFIX);

            $status = $bodyData['payment']['status'];

            $order = wc_get_order($order_id);
            if (!$order) {
                http_response_code('404');
                die('order not found');
            }

            if ($status === 'success') {
                $order->payment_complete($bodyData->operation->id);
                $order->add_order_note(__('Payment processed. Psphost.'));
                die('ok');
            }
            $order->update_status('failed');
            $order->add_order_note(__('Payment failed. Psphost.'));
            die('fail');
        }
        http_response_code(400);
        die('invalid signature');
    }

    public function process_success($order_id) {
        $test = $this->get_option('test') === 'yes';
        if ($test) {
            $order = new WC_Order($order_id);
            $order->payment_complete();
            wp_redirect($order->get_view_order_url());
            exit;
        }
    }

    public function process_refund($order_id, $amount = null, $reason = '')
    {
        return true;
    }

    /**
     * @param WC_Order_Refund $refund
     * @param array $args
     * @throws Exception
     */
    public function process_just_created_refund($refund, $args)
    {
        if (!$args['refund_payment']) {
            return;
        }

        $order = WC_Order_Factory::get_order($refund->get_parent_id());

        if ($order === false) {
            throw new Exception('Unable to find a parent order for the specified refund.');
        }

        if (!in_array($order->get_status(), array('processing', 'completed', 'pt-refunded'))) {
            throw new Exception('Inappropriate order status. It should be "processing" or "complete" or "pt-refunded".');
        }

        try {
            $psphost_refund_result = $this->get_refund_processor()->processRefund(
                (string)$order->get_id(),
                $refund->get_amount(),
                $order->get_currency(),
                $refund->get_reason()
            );
        } catch (Exception $e) {
            throw new Exception('Request was not correctly processed by gateway.');
        }

        if ($psphost_refund_result->getRefundExternalId() === null) {
            throw new Exception('Request was declined by gateway.');
        }

        $refund->add_meta_data('_gateway_operation_id', $psphost_refund_result->getRefundExternalId(), true);
    }

    /**
     * @param int $refund_id
     * @param array $args
     */
    public static function process_just_saved_refund($refund_id, $args)
    {
        if (!$args['refund_payment']) {
            return;
        }
        $now = new DateTime();
        $comment = 'Money-back request #' . $refund_id . ' was sent at ' . $now->format('d.m.Y H:i:s');

        self::extend_refund_comment($refund_id, $comment);

        $refund = WC_Order_Factory::get_order($refund_id);
        $order_id = $refund->get_parent_id();

        $comment_id = self::append_order_comment($order_id, $comment);

        if ($comment_id) {
            update_post_meta($refund_id, '_refund_request_comment_id', $comment_id);
        }
    }

    /**
     * @param int $refund_id
     * @param string $extension
     */
    private static function extend_refund_comment($refund_id, $extension)
    {
        $comment = get_post_meta($refund_id, '_refund_reason', true);
        if (empty($comment)) {
            $comment = $extension;
        } else {
            $comment .= ' | ' . $extension;
        }
        update_post_meta($refund_id, '_refund_reason', $comment);
    }

    /**
     * @param int $order_id
     * @param string $comment
     * @param int $parent_comment
     * @return int|null
     */
    private static function append_order_comment($order_id, $comment, $parent_comment = 0)
    {
        $commentData = array(
            'comment_post_ID'      => $order_id,
            'comment_author'       => 'Psphost',
            'comment_agent'        => 'Psphost',
            'comment_author_email' => '',
            'comment_author_url'   => 'https://psphost.ru',
            'comment_content'      => $comment,
            'comment_type'         => 'order_note',
            'comment_approved'     => 1,
            'comment_parent'       => $parent_comment,
            'user_id'              => 0,
        );

        $result = wp_insert_comment($commentData);
        if (!is_numeric($result)) {
            return null;
        }
        return $result;
    }

    /**
     * @param array $data
     */
    public function process_refund_callback($data)
    {
        try {
            $psphost_callback_result = $this->get_refund_processor()->processCallback($data);
        } catch (Exception $e) {
            if ($e instanceof PsphostOperationException) {
                return;
            }
            http_response_code(400);
            die($e->getMessage());
        }

        global $wpdb;
        $results = $wpdb->get_results("
            SELECT meta_id, post_id FROM {$wpdb->prefix}postmeta
            WHERE meta_key = '_gateway_operation_id' AND meta_value = '{$psphost_callback_result->getRefundExternalId()}'
        ", ARRAY_A);

        if (empty($results)) {
            http_response_code(400);
            die('Unknown operation identifier');
        }

        $meta_id = $results[0]['meta_id'];
        $refund_id = $results[0]['post_id'];

        $now = new DateTime();
        if ($psphost_callback_result->isSuccess()) {
            $comment = 'Money-back request #' . $refund_id . ' was successfully processed at ' . $now->format('d.m.Y H:i:s');
        } else {
            $comment = 'Money-back request #' . $refund_id . ' was declined at ' . $now->format('d.m.Y H:i:s') . '. Reason: ' . $psphost_callback_result->getDescription();
        }

        self::extend_refund_comment($refund_id, $comment);

        $refund_request_comment_id = get_post_meta($refund_id, '_refund_request_comment_id', true);
        if ($refund_request_comment_id) {
            self::append_order_comment($psphost_callback_result->getOrderId(), $comment, $refund_request_comment_id);
        }

        $wpdb->query("UPDATE {$wpdb->prefix}postmeta SET meta_value = '+{$psphost_callback_result->getRefundExternalId()}' WHERE meta_id = '{$meta_id}'");

        $order = wc_get_order($psphost_callback_result->getOrderId());

        if($psphost_callback_result->getPaymentStatus() === 'partially refunded'){
            $order->update_status('pt-refunded');
        } else {
            $order->update_status('refunded');
        }
        http_response_code(200);
        die();
    }

    /**
     * @return PsphostRefundProcessor
     */
    public function get_refund_processor()
    {
        return $this->refund_processor;
    }
}
